/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx_open_endpoint.c,v 1.40 2005/07/15 14:43:04 gallatin Exp $";

#include "mx_auto_config.h"
#include <sys/types.h>
#include <sys/socket.h>
#if MX_OS_LINUX || MX_OS_UDRV
#include <linux/if.h>
#include <linux/sockios.h>
#elif MX_OS_FREEBSD || MX_OS_MACOSX || MX_OS_SOLARIS
#include <net/if.h>
#include <sys/sockio.h>
#endif
#include <netinet/in.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>

#include "mx_auto_config.h"
#include "myriexpress.h"
#include "internal.h"
#include "attrib.h"
#include "tcp.h"

struct mx_endpoint *mx_new_endpoint(void);
void mx_destroy_endpoint(struct mx_endpoint *);
mx_return_t mx_get_my_address(struct mx_endpoint *, int, uint32_t);

mx_return_t
mx_open_endpoint(uint32_t board_num, 
                 uint32_t endpoint_id,
		 uint32_t filter,
		 mx_param_t *params_list, 
                 uint32_t params_count,
		 mx_endpoint_t *endpoint)
{
    struct mx_endpoint *new_endpoint;
    mx_return_t rc;
    uint32_t endpoint_num;
    struct sockaddr_in saddr;
    
    pthread_mutex_lock(&Mx_po_lock);
    
    /* allocate a new endpoint structure */
    new_endpoint = mx_new_endpoint();
    if (new_endpoint == NULL) {
        pthread_mutex_unlock(&Mx_po_lock);
        return MX_NO_RESOURCES;
    }
    
    /*
     * Our "endpoint" is implemented as a TCP socket.  We will listen on 
     * TT_BASE_ENDPOINT + endpoint_num, where endpoint_num is the mx endpoint number.
     * We find our endpoint_num by opening a socket, then successively attempting
     * binds until it works.  That will be our endpoint number.
     */
    new_endpoint->socket = socket(PF_INET, SOCK_STREAM, IPPROTO_IP);
    if (new_endpoint->socket == -1) {
        mx_destroy_endpoint(new_endpoint);
        pthread_mutex_unlock(&Mx_po_lock);
        return MX_NO_RESOURCES;
    }
    MX_DEBUG_PRINT(MX_DEBUG_TCP_LIB,("my listening socket is %d\n", new_endpoint->socket));
    
    /* XXX - set O_NONBLOCK on socket */
    
    /* start building out address info */
    saddr.sin_family = AF_INET;
    saddr.sin_addr.s_addr = INADDR_ANY;
    
    for (endpoint_num = 1; endpoint_num < MX_MAX_ENDPOINT; ++endpoint_num) {
        saddr.sin_port = htons(endpoint_num + ENDPOINT_BASE);
        if (bind(new_endpoint->socket, (struct sockaddr *)&saddr,
                 sizeof (saddr)) == 0) {
            break;
        }
    }
    
    /* include this socket in select() set for this endpoint */
    mx_endpoint_sock_insert(new_endpoint, new_endpoint->socket);

    /* if endpoint_num >= MX_MAX_ENDPOINT, we failed to get a endpoint number */
    if (endpoint_num >= MX_MAX_ENDPOINT) {
        mx_destroy_endpoint(new_endpoint);
        pthread_mutex_unlock(&Mx_po_lock);
        return MX_NO_RESOURCES;
    }
    MX_DEBUG_PRINT(MX_DEBUG_TCP_LIB,("successful, endpoint ID = %d\n", 
              ntohs(saddr.sin_port) - ENDPOINT_BASE));
    rc = listen(new_endpoint->socket, 16);
    if (rc == -1) {
        perror("listen");
        mx_destroy_endpoint(new_endpoint);
        pthread_mutex_unlock(&Mx_po_lock);
        return MX_NO_RESOURCES;
    }
    
    /* build our address */
    rc = mx_get_my_address(new_endpoint, endpoint_num, filter);
    if (rc != MX_SUCCESS) {
        mx_destroy_endpoint(new_endpoint);
        pthread_mutex_unlock(&Mx_po_lock);
        return (rc);
    }
    
    /* link this endpoint into Global endpoint list */
    MX_LIST_INSERT(&Mx_endpoints, new_endpoint);
    
    /* Set up default attributes for this endpoint */
    rc = mx_set_dflt_endpoint_attribs(new_endpoint);
    if (rc != MX_SUCCESS) {
        mx_destroy_endpoint(new_endpoint);
        return (rc);
    }
    
    /*
     * At this point, we have a structure and a endpoint we can talk on,
     * time to return it to the user and let him start playing.
     */
    *endpoint = (mx_endpoint_t)new_endpoint;
    pthread_mutex_unlock(&Mx_po_lock);
    return MX_SUCCESS;
}

struct mx_endpoint *
mx_new_endpoint()
{
    struct mx_endpoint *pp;
    
    pp = (struct mx_endpoint *) calloc(1, sizeof (*pp));
    if (pp == NULL) {
        return (NULL);
    }
    
    /* initialize circular linked lists */
    MX_LIST_INIT(&pp->recv_list);
    MX_LIST_INIT(&pp->send_list);
    MX_LIST_INIT(&pp->bcast_list);
    MX_LIST_INIT(&pp->barrier_list);
    MX_LIST_INIT(&pp->completed_posts);
    MX_LIST_INIT(&pp->buffered_posts);
    MX_LIST_INIT(&pp->putget_posts);
    MX_LIST_INIT(&pp->pending_barriers);
    MX_LIST_INIT(&pp->queued_messages);
    MX_LIST_INIT(&pp->known_addresss);

    pthread_cond_init(&pp->peek_cond, NULL);

    /* initialize some fields */
    pp->socket = -1;
    mx_endpoint_sock_init(pp);
    return (pp);
}

void
mx_destroy_endpoint(struct mx_endpoint *endpoint)
{
    /* close any sockets we have open */
    if (endpoint->socket >= 0) {
        close(endpoint->socket);
    }
    
    /* free all posts associated with this endpoint */
    MX_LIST_APPLY(struct mx_post, &endpoint->recv_list, mx_destroy_post);
    MX_LIST_APPLY(struct mx_post, &endpoint->send_list, mx_destroy_post);
    MX_LIST_APPLY(struct mx_post, &endpoint->bcast_list, mx_destroy_post);
    MX_LIST_APPLY(struct mx_post, &endpoint->completed_posts, mx_destroy_post);
    MX_LIST_APPLY(struct mx_post, &endpoint->buffered_posts, mx_destroy_post);
    MX_LIST_APPLY(struct mx_post, &endpoint->putget_posts, mx_destroy_post);

    /* free all queued messages */
    MX_LIST_APPLY(struct mx_queued_msg, &endpoint->queued_messages, free);
    
    /* free all known addresss */
    MX_LIST_APPLY(struct mx_address_desc, &endpoint->known_addresss,
                  mx_destroy_address_desc);

    pthread_cond_destroy(&endpoint->peek_cond);
    /* pthread_mutex_destroy(&endpoint->lock); */
    MX_LIST_REMOVE(endpoint);
    mx_free_attr_list(endpoint);
    free(endpoint);
}

mx_return_t
mx_get_my_address(struct mx_endpoint *endpoint,
                  int endpoint_num, 
                  uint32_t filter)
{
    int s;
    struct sockaddr_in *sa;
    struct ifreq ifr;
    struct mx_lib_address *endp;
    char *ifname[] = { "eth0", "eth1", "fxp0", "lo", NULL };
    int i;
    
    /* access local address struct */
    endp = (struct mx_lib_address *)&endpoint->my_address;
    
    s = socket(PF_INET, SOCK_DGRAM, IPPROTO_IP);
    if (s == -1) {
        perror("socket");
        return (MX_BAD_BAD_BAD);
    }
    
    /* Cycle through list of interface names until we find one that we like */
    /* XXX interface name list should be generated programmatically */
    for (i=0; ifname[i] != NULL; ++i) {
        strcpy(ifr.ifr_name, ifname[i]);
        if (ioctl(s, SIOCGIFADDR, &ifr) == -1) {
            perror(ifname[i]);
        } else {
            printf("Using interface \"%s\"\n", ifname[i]);
            break;
        }
    }
    
    /* see if we found a name */
    if (ifname[i] == NULL) {
        fprintf(stderr, "Unable to determine interface name!\n");
        return (MX_BAD_BAD_BAD);
    }
    
    sa = (struct sockaddr_in *)&(ifr.ifr_addr);
    
    /* copy IP info into address */
    memcpy(&endp->ipaddr_n, &sa->sin_addr, sizeof (endp->ipaddr_n));
    endp->ipport_n = htons(endpoint_num + ENDPOINT_BASE);
    endp->filter_n = htonl(filter);
    close (s);
    
    return MX_SUCCESS;
}
